package kz.arta.ext.sap.service;

import org.testng.annotations.Test;

import kz.arta.ext.sap.db.UserManager;

import static org.mockito.Mockito.*;
import static org.testng.Assert.assertEquals;

import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.MockitoAnnotations;

public class PasswordSetServiceTest {

	@Mock private UserManager um;
	
	@Test
	public void errorMessagesHandlerTest() throws Exception {
		assertEquals("{\"result\":\"success\", \"errorCode\":\"0\"}", PasswordSetService.errorMessagesHandler(200));
		assertEquals("{\"status\":\"Something has gone wrong on server/db\", \"errorCode\": \"500\"}", PasswordSetService.errorMessagesHandler(111));
		assertEquals("{\"status\":\"401 Unauthorized!\", \"errorCode\": \"401\"}", PasswordSetService.errorMessagesHandler(401));
		assertEquals("{\"status\":\"403 Forbidden.\", \"error\":\"You don't have any access to the requested account\", \"errorCode\": \"403\"}", PasswordSetService.errorMessagesHandler(403));
		assertEquals("{\"status\":\"There is no user with such login!\", \"errorCode\": \"404\"}", PasswordSetService.errorMessagesHandler(404));
	}

	/**
	 * Тест пользователя, который меняет пароль к чужой записи
	 * @throws 
	 */
	@Test
	public void mockSetPassword_accessForbidden() throws Exception {
		SynergyUser suAuth = new SynergyUser();
		suAuth.setAuth(true);
		
		MockitoAnnotations.initMocks(this);
		doReturn(suAuth).when(um).selectFromUsersQuery(anyString(), anyString(), anyString());
		
		PasswordSetService pass = new PasswordSetService(um);
		assertEquals("{\"status\":\"403 Forbidden.\", \"error\":\"You don't have any access to the requested account\", \"errorCode\": \"403\"}", pass.setPassword("login", "pass", "pass2", "log2", "new_pass"));
		verify(um).selectFromUsersQuery(anyString(), anyString(), anyString());
	}
	
	/**
	 * Тест пользователя, который меняет пароль к своей записи
	 * @throws 
	 */
	@Test
	public void mockSetPassword_userSuccess() throws Exception {
		SynergyUser suAuth = new SynergyUser();
		suAuth.setAuth(true);
		
		MockitoAnnotations.initMocks(this);
		doReturn(suAuth).when(um).selectFromUsersQuery(anyString(), anyString(), anyString());
		Mockito.when(um.setPass(anyString(), anyString())).thenReturn(200);
				
		PasswordSetService pass = new PasswordSetService(um);
		assertEquals("{\"result\":\"success\", \"errorCode\":\"0\"}", pass.setPassword("pass", "pass", "pass", "pass", "new_pass"));
		verify(um).selectFromUsersQuery(anyString(), anyString(), anyString());
		verify(um).setPass(anyString(), anyString());
	}
	
	
	/**
	 * Тест админа, который меняет пароль к записи существующего пользователя
	 * @throws 
	 */
	@Test
	public void mockSetPassword_successFromAdmin() throws Exception {
		SynergyUser synUser = new SynergyUser();
		synUser.setAdmin(true);
		
		MockitoAnnotations.initMocks(this);
		Mockito.when(um.selectFromUsersQuery(anyString(), anyString(), anyString())).thenReturn(synUser);
		Mockito.when(um.setPass(anyString(), anyString())).thenReturn(200);
		
		PasswordSetService pss = new PasswordSetService(um);
		assertEquals(pss.setPassword("frst", "second", "third", "frth", "fifth"), "{\"result\":\"success\", \"errorCode\":\"0\"}");
		verify(um).selectFromUsersQuery(anyString(), anyString(), anyString());
		verify(um).setPass(anyString(), anyString());
	}
	
	/**
	 * Тест админа, который меняет пароль к записи несуществующего пользователя
	 * @throws 
	 */
	@Test
	public void mockSetPassword_failFromAdmin() throws Exception {
		SynergyUser synUser = new SynergyUser();
		synUser.setAdmin(true);
		
		MockitoAnnotations.initMocks(this);
		Mockito.when(um.selectFromUsersQuery(anyString(), anyString(), anyString())).thenReturn(synUser);
		Mockito.when(um.setPass(anyString(), anyString())).thenReturn(404);
		
		PasswordSetService pss = new PasswordSetService(um);
		assertEquals(pss.setPassword("frst", "second", "third", "frth", "fifth"), "{\"status\":\"There is no user with such login!\", \"errorCode\": \"404\"}");
		verify(um).selectFromUsersQuery(anyString(), anyString(), anyString());
		verify(um).setPass(anyString(), anyString());
	}
	
  /*
	@Test
	public void mockTest_success() throws Exception {
		PasswordSetService pss = new PasswordSetService();
		assertEquals("{\"status\":\"working\"}", pss.test());
	}
  */
	
}


