package kz.arta.test.blocking.data;

import org.codehaus.jackson.annotate.JsonIgnore;
import org.codehaus.jackson.annotate.JsonIgnoreProperties;
import org.codehaus.jackson.map.annotate.JsonSerialize;

import java.io.Serializable;
import java.util.*;

/**
 * Класс предназначен для сериализации/десериализации данных форм
 *
 * @author val
 * @since 04.02.2013 09:51
 */
@JsonIgnoreProperties(ignoreUnknown = true)
public class ASFData implements Serializable {

    private List<Data> data = new ArrayList<Data>();

    /**
     * Внутренний класс предназначенный для работы с атомарной единицей данных формы (данными одного виджета)
     */
    @JsonIgnoreProperties(ignoreUnknown = true)
    public static class Data implements Serializable {

        /**
         * идентификатор компонента которому принадлежат данные
         */
        private String id;

        /**
         * тип компонента которому принадлежат данные. Тип должен коррелировать c {@link WidgetType}
         */
        private String type;
        /**
         * надпись компонента которому принадлежат данные
         */
        @JsonSerialize(include = JsonSerialize.Inclusion.NON_NULL)
        private String label;
        /**
         * значение компонента которому принадлежат данные
         */
        @JsonSerialize(include = JsonSerialize.Inclusion.NON_NULL)
        private String value;
        /**
         * ключ компонента которому принадлежат данные
         */
        @JsonSerialize(include = JsonSerialize.Inclusion.NON_NULL)
        private String key;
        /**
         * идентификатор значения компонента которому принадлежат данные
         */
        @JsonSerialize(include = JsonSerialize.Inclusion.NON_NULL)
        private String valueID;
        /**
         * имя пользователя используемого в компоненте которому принадлежат данные
         */
        @JsonSerialize(include = JsonSerialize.Inclusion.NON_NULL)
        private String username;
        /**
         * идентификатор пользователя используемого в компоненте которому принадлежат данные
         */
        @JsonSerialize(include = JsonSerialize.Inclusion.NON_NULL)
        private String userID;
        /**
         * Список значений компонента которому принадлежат данные
         */
        @JsonSerialize(include = JsonSerialize.Inclusion.NON_NULL)
        private List<String> values;
        /**
         * Список ключей компонента которому принадлежат данные
         */
        @JsonSerialize(include = JsonSerialize.Inclusion.NON_NULL)
        private List<String> keys;
        /**
         * Список вложенных данных компонента которому принадлежат данные. Применимо только к динамическим таблицам
         */
        @JsonSerialize(include = JsonSerialize.Inclusion.NON_EMPTY)
        private List<Data> data;

        /**
         * Версия формата данных
         */
        @JsonSerialize(include = JsonSerialize.Inclusion.NON_NULL)
        private String formatVersion;

        /**
         * Карта из id пользователя и присвоенного ему названия в компоненте выбора пользователей
         */
        @JsonSerialize(include = JsonSerialize.Inclusion.NON_NULL)
        private Map<String, String> manualTags;

        public Data() {
        }

        public String getFormatVersion() {
            return formatVersion;
        }

        public void setFormatVersion(String formatVersion) {
            this.formatVersion = formatVersion;
        }

        public Map<String, String> getManualTags() {
            return manualTags;
        }

        public void setManualTags(Map<String, String> manualTags) {
            this.manualTags = manualTags;
        }

        public String getId() {
            return id;
        }

        public Data setId(String id) {
            this.id = id;
            return this;
        }

        public String getType() {
            return type;
        }

        public void setType(String type) {
            this.type = type;
        }

        public String getLabel() {
            return label;
        }

        public void setLabel(String label) {
            this.label = label;
        }

        public String getValue() {
            return value;
        }

        public Data setValue(String value) {
            this.value = value;
            return this;
        }

        public String getKey() {
            return key;
        }

        public void setKey(String key) {
            this.key = key;
        }

        public String getValueID() {
            return valueID;
        }

        public void setValueID(String valueID) {
            this.valueID = valueID;
        }

        public String getUsername() {
            return username;
        }

        public void setUsername(String username) {
            this.username = username;
        }

        public String getUserID() {
            return userID;
        }

        public void setUserID(String userID) {
            this.userID = userID;
        }

        public List<String> getValues() {
            return values;
        }

        public void setValues(List<String> values) {
            this.values = values;
        }

        public List<String> getKeys() {
            return keys;
        }

        public void setKeys(List<String> keys) {
            this.keys = keys;
        }

        public List<Data> getData() {
            return data;
        }

        @JsonIgnore
        public List<Data> getDataCheckNull() {
            if (data == null) {
                data = new ArrayList<>();
            }
            return data;
        }

        public void setData(List<Data> data) {
            this.data = data;
        }

        /**
         * Метод получение данных по идентификатору компонента
         *
         * @param cmpID - идентификатор компонента
         * @return - данные компонента. если компонент не найден возвращается null
         */
        public Data getData(String cmpID) {
            if (data != null) {
                for (Data d : data) {
                    if (d.getId().equalsIgnoreCase(cmpID)) {
                        return d;
                    }
                }
            }
            return null;
        }

        /**
         * @return множество блоков данных в дин.таблице в виде ["b1","b2","b3"]
         */
        @JsonIgnore
        public Set<String> getBlockSet() {
            if (data != null) {
                Set<String> set = new LinkedHashSet<String>();
                for (Data d : data) {
                    if (d!=null&&d.getId().lastIndexOf("-b") > 0) {
                        set.add(d.getId().substring(d.getId().lastIndexOf("-b") + 1));
                    }
                }
                return set;
            }
            return null;
        }

        /**
         * @return последний номер блока данных в динамической таблице
         */
        @JsonIgnore
        public int getLastBlockNumber() {
            int lastBlockNumber = 0;
            Set<String> blocks = getBlockSet();
            if (blocks != null) {
                for (String block : blocks) {
                    int blockNum = 0;
                    try {
                        blockNum = Integer.parseInt(block.substring(1).trim());
                    } catch (NumberFormatException ignored) {
                    }
                    if (blockNum > lastBlockNumber) {
                        lastBlockNumber = blockNum;
                    }
                }
            }
            return lastBlockNumber;
        }

        public void addData(Data data) {

            WidgetType type = WidgetType.getFromString(data.getType());
            if(type == null) {
                throw new IllegalArgumentException("unknown widget type \""+data.getType()+"\" in component with id \""+data.getId()+"\"");
            }

            getDataCheckNull().add(data);
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;
            Data data1 = (Data) o;
            return Objects.equals(id, data1.id) &&
                    Objects.equals(type, data1.type) &&
                    Objects.equals(label, data1.label) &&
                    Objects.equals(value, data1.value) &&
                    Objects.equals(key, data1.key) &&
                    Objects.equals(valueID, data1.valueID) &&
                    Objects.equals(username, data1.username) &&
                    Objects.equals(userID, data1.userID) &&
                    Objects.equals(values, data1.values) &&
                    Objects.equals(keys, data1.keys) &&
                    Objects.equals(data, data1.data) &&
                    Objects.equals(formatVersion, data1.formatVersion) &&
                    Objects.equals(manualTags, data1.manualTags);
        }

        @Override
        public int hashCode() {
            return Objects.hash(id, type, label, value, key, valueID, username, userID, values, keys, data, formatVersion, manualTags);
        }
    }

    public List<Data> getData() {
        return data;
    }

    public Data getData(String cmpID) {
        for (Data data : this.data) {
            if (data != null && data.getId() != null && data.getId().equals(cmpID)) {
                return data;
            }
        }
        return null;
    }

    public void setData(List<Data> data) {
        this.data = data;
    }

    public void addData(Data data) {

        WidgetType type = WidgetType.getFromString(data.getType());
        if(type == null) {
            throw new IllegalArgumentException("unknown widget type \""+data.getType()+"\" in component with id \""+data.getId()+"\"");
        }

        removeData(data.getId());
        this.data.add(data);
    }

    public void removeData(String cmpID) {
        if(cmpID == null) return;
        Iterator<Data> iterator = this.getData().iterator();
        while(iterator.hasNext()) {
            Data d = iterator.next();
            if(d != null && cmpID.equals(d.getId())) {
                iterator.remove();
            }
        }
    }
}
