package kz.arta.test.blocking.utils;

import com.google.common.io.ByteStreams;
import com.google.common.primitives.Ints;
import org.apache.commons.io.IOUtils;
import org.codehaus.jackson.JsonNode;
import org.codehaus.jackson.map.ObjectMapper;

import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.Map;

/**
 * Created by exile
 * Date: 10.01.17
 * Time: 12:12
 */
public class HttpBasicOperation {
    public static ObjectMapper mapper = new ObjectMapper();

    public HttpURLConnection openPostConnection(URL url) throws IOException {
        return openPostConnection(url, null);
    }

    public HttpURLConnection openPostConnection(URL url, String auth) throws IOException {
        return openConnection(url, "POST", auth, null, null);
    }

    public HttpURLConnection openPostConnection(URL url, String auth, String accept) throws IOException {
        return openConnection(url, "POST", auth, accept, null);
    }

    public HttpURLConnection openPostConnection(URL url, String auth, String accept, String contentType) throws IOException {
        return openConnection(url, "POST", auth, accept, contentType);
    }

    public HttpURLConnection openDeleteConnection(URL url) throws  IOException{
        return openConnection(url, "DELETE", null, null, null);
    }

    public HttpURLConnection openGetConnection(URL url) throws  IOException{
        return openGetConnection(url, null);
    }

    public HttpURLConnection openGetConnection(URL url, String auth) throws IOException {
        return openGetConnection(url, auth, null);
    }

    public HttpURLConnection openGetConnection(URL url, String auth, String accept) throws IOException {
        return openConnection(url, "GET", auth, accept, null);
    }

    public HttpURLConnection openHeadConnection(URL url) throws  IOException{
        return openConnection(url, "HEAD", null, null, null);
    }

    public HttpURLConnection openPutConnection(URL url) throws IOException{
        return openConnection(url, "PUT", null, null, null);
    }

    private HttpURLConnection openConnection(URL url, String method, String auth, String accept, String contentType) throws IOException{
        HttpURLConnection conn = (HttpURLConnection) url.openConnection();
        conn.setRequestMethod(method);
        if (method.equals("POST")) {
            conn.setDoOutput(true);
        }
        addHeaders(conn, auth, accept, contentType);
        return conn;
    }

    private void addHeaders(HttpURLConnection conn, String auth, String accept, String contentType){
        if (contentType == null) {
            conn.setRequestProperty("Content-Type", "application/json; charset=utf-8");
        } else {
            conn.setRequestProperty("Content-Type", contentType);
        }
        if (accept == null) {
            conn.setRequestProperty("Accept", "application/json; charset=utf-8");
        } else {
            conn.setRequestProperty("Accept", accept);
        }
        if (auth != null) {
            conn.setRequestProperty("Authorization", auth);
        }
    }

    public void writeValue(HttpURLConnection connection, Object value) throws IOException{
        connection.setDoOutput(true);
        try (OutputStream stream = connection.getOutputStream()) {
            mapper.writeValue(stream, value);
        } catch (IOException e) {
            connection.disconnect();
            throw e;
        }

    }

    public void writeParams(HttpURLConnection connection, Map<String, String> params) throws IOException{
        connection.setDoOutput(true);

        DataOutputStream request = new DataOutputStream(connection.getOutputStream());
        for(String key : params.keySet()){
            request.writeBytes(key+"=" + URLEncoder.encode(params.get(key), StandardCharsets.UTF_8.name()));
            request.writeBytes("&");
        }
    }

    public String readResult(HttpURLConnection conn) throws IOException{
        try {
            InputStream in = conn.getInputStream();
            return IOUtils.toString(in, StandardCharsets.UTF_8.name());
        } catch (IOException e) {
            throw new IOException(readError(conn), e);
        } finally {
            conn.disconnect();
        }
    }

    public JsonNode readJsonResult(HttpURLConnection conn) throws IOException{
        try {

            return mapper.readTree(conn.getInputStream());
        } catch (IOException e) {
            throw new IOException(readError(conn), e);
        } finally {
            conn.disconnect();
        }
    }


    public <T> T readResult(HttpURLConnection conn, Class<T> resultClass, T httpErrorCodeInstance, int... allowedStatusCodes) throws IOException{
        try {
            return mapper.readValue(conn.getInputStream(), resultClass);
        } catch (IOException e) {
            int code = conn.getResponseCode();
            if(allowedStatusCodes != null) {
                if(Ints.contains(allowedStatusCodes, code)) {
                    return httpErrorCodeInstance;
                }
            }
            throw new IOException(readError(conn), e);
        } finally {
            conn.disconnect();
        }
    }

    public <T> T readResult(HttpURLConnection conn, Class<T> resultClass) throws IOException{
        try {
            InputStream in = conn.getInputStream();
            byte[] b = ByteStreams.toByteArray(in);
            if (b.length == 0) {
                return null;
            }
            return mapper.readValue(b, resultClass);
        } catch (IOException e) {
            throw new IOException(readError(conn), e);
        } finally {
            conn.disconnect();
        }
    }

    private String readError(HttpURLConnection conn){
        try {
            InputStream in = conn.getErrorStream();
            byte[] b = ByteStreams.toByteArray(in);
            return new String(b, StandardCharsets.UTF_8);
        } catch (Exception e) {
            return e.getMessage();
        }
    }
}
