package kz.arta.synergy.api.asforms;

import kz.arta.synergy.api.JsonUtils;
import kz.arta.synergy.api.Query;
import kz.arta.synergy.api.QueryContext;
import kz.arta.synergy.api.RestHttpQuery;
import kz.arta.synergy.api.asforms.converter.AsFormConverter;
import kz.arta.synergy.api.asforms.converter.DefaultAsFormConverter;
import kz.arta.synergy.api.asforms.exceptions.FormRecordNotFound;
import kz.arta.synergy.api.asforms.exceptions.FoundTooManyRecords;
import kz.arta.synergy.api.asforms.pojo.*;
import kz.arta.synergy.api.asforms.utils.SearchIndexParamsBuilder;
import org.codehaus.jackson.type.TypeReference;

import java.io.IOException;
import java.util.List;

/**
 * @author raimbek
 * @since 09.11.2016
 */
public class AsFormService {

    private RestHttpQuery restHttpQuery;
    private AsFormConverter asFormConverter;

    private AsFormService(QueryContext context) {
        this.restHttpQuery = new RestHttpQuery(context);
    }

    public static AsFormService newInstance(QueryContext context) {
        AsFormService asFormService = new AsFormService(context);
        asFormService.asFormConverter = new DefaultAsFormConverter();
        return asFormService;
    }

    public RestHttpQuery getRestHttpQuery() {
        return restHttpQuery;
    }

    public AsFormService setRestHttpQuery(RestHttpQuery restHttpQuery) {
        this.restHttpQuery = restHttpQuery;
        return this;
    }

    public AsFormService setFormConverter(AsFormConverter converter) {
        this.asFormConverter = converter;
        return this;
    }

    public AsFormConverter getAsFormConverter() {
        return asFormConverter;
    }

    public List<AdvancedSearchResult> advancedSearch(AdvancedSearchParams searchParams) throws IOException {
        Query query = Query.newInstance()
                .methodPost()
                .header("Content-Type", "application/json; charset=utf-8")
                .url("/rest/api/asforms/search/advanced")
                .body(JsonUtils.toJson(searchParams));

        String result = restHttpQuery.doQuery(query);
        return JsonUtils.read(result, new TypeReference<List<AdvancedSearchResult>>() {});
    }

    public AdvancedSearchResult advancedSearchOne(AdvancedSearchParams searchParams)
            throws IOException, FormRecordNotFound, FoundTooManyRecords
    {
        Query query = Query.newInstance()
                .methodPost()
                .header("Content-Type", "application/json; charset=utf-8")
                .url("/rest/api/asforms/search/advanced")
                .body(JsonUtils.toJson(searchParams));

        String response = restHttpQuery.doQuery(query);
        List<AdvancedSearchResult> results = JsonUtils.read(response, new TypeReference<List<AdvancedSearchResult>>() {});

        if (results == null || results.isEmpty()) {
            throw new FormRecordNotFound();
        } else if (results.size() > 1) {
            throw new FoundTooManyRecords();
        }
        return results.get(0);
    }

    public List<AdvancedSearchResult> advancedSearch(String formId, String key, String value) throws IOException {
        return advancedSearch(AdvancedSearchParams.build(
            String.format("where uuid='%s' and %s='%s'", formId, key, value), key
        ));
    }

    public AdvancedSearchResult advancedSearchOne(String formId, String key, String value) throws IOException, FormRecordNotFound, FoundTooManyRecords {
        return advancedSearchOne(AdvancedSearchParams.build(
                String.format("where uuid='%s' and %s='%s'", formId, key, value), key
        ));
    }

    public SearchIndexResult searchIndex(SearchIndexParamsBuilder builder) throws IOException {
        Query query = Query.newInstance().url("/rest/api/asforms/search_index?" + builder.build());
        String response = restHttpQuery.doQuery(query);
        return JsonUtils.read(response, SearchIndexResult.class);
    }

    public RegistryRecord searchIndexOne(SearchIndexParamsBuilder builder) throws IOException, FormRecordNotFound, FoundTooManyRecords {
        SearchIndexResult result = searchIndex(builder);
        if (result == null || result.getCount() == 0) {
            throw new FormRecordNotFound(builder.build());
        } else if (result.getCount() > 1) {
            throw new FoundTooManyRecords(builder.build());
        }
        return result.getRecords().get(0);
    }

    public SearchIndexResult searchIndex(String formId, String key, String value) throws IOException {
        SearchIndexParamsBuilder builder = SearchIndexParamsBuilder.newBuilder()
                .form(formId)
                .exact(key, value);
        return searchIndex(builder);
    }

    public RegistryRecord searchIndexOne(String formId, String key, String value) throws IOException, FormRecordNotFound, FoundTooManyRecords {
        SearchIndexParamsBuilder builder = SearchIndexParamsBuilder.newBuilder()
                .form(formId)
                .exact(key, value);
        return searchIndexOne(builder);
    }

    public <T extends AsForm> T getData(Class<T> formClass, String dataUUID) throws IOException {
        return asFormConverter.toAsForm(formClass, getData(dataUUID));
    }

    public AsFormWrapper getData(String dataUUID) throws IOException {
        String data = restHttpQuery.doQuery(Query.newInstance().url("/rest/api/asforms/data/" + dataUUID));
        return JsonUtils.read(data, AsFormWrapper.class);
    }

    public String saveData(AsFormWrapper asFormWrapper) throws IOException {
        Query query = Query.newInstance()
                .methodPost()
                .url("/rest/api/asforms/data/save")
                .formParam("formUUID", asFormWrapper.getForm())
                .formParam("uuid", asFormWrapper.getUuid())
                .formParam("data", "\"data\":" + JsonUtils.toJson(asFormWrapper.getData()));

        return restHttpQuery.doQuery(query);
    }

    public <T extends AsForm> String saveData(T asForm) throws IOException {
        AsFormWrapper asFormWrapper = asFormConverter.toAsfData(asForm);
        return saveData(asFormWrapper);
    }

    public <T extends AsForm> AsFormWrapper toAsfData(T asForm) {
        return asFormConverter.toAsfData(asForm);
    }

    public byte[] printForm(String dataUUID, String format) throws IOException {
        Query query = Query.newInstance()
                .url("/rest/api/asforms/template/print/form")
                .queryParam("dataUUID", dataUUID)
                .queryParam("format", format);

        return restHttpQuery.doQueryAndReturnBytes(query);
    }

    public AsFormDefinition getFormDefinitionExt(String formId, String formCode, String version) throws IOException {
        Query query = Query.newInstance()
                .url("/rest/api/asforms/form_ext")
                .queryParam("formID", formId)
                .queryParam("formCode", formCode)
                .queryParam("version", version);
        String result = restHttpQuery.doQuery(query);
        return JsonUtils.read(result, AsFormDefinition.class);
    }

    public String getDocumentIdByDataUuid(String dataUuid) throws IOException {
        Query query = Query.newInstance()
                .url("/rest/api/asforms/data/document")
                .queryParam("dataUUID", dataUuid);
        String result = restHttpQuery.doQuery(query);
        return JsonUtils.getValueByPath(result, dataUuid + ".documentID");
    }
}
