package kz.arta.synergy.api.asforms;

import kz.arta.synergy.api.JsonUtils;
import kz.arta.synergy.api.Query;
import kz.arta.synergy.api.QueryContext;
import kz.arta.synergy.api.RestHttpQuery;
import kz.arta.synergy.api.asforms.annotations.*;
import kz.arta.synergy.api.asforms.exceptions.CreateAsFormException;
import kz.arta.synergy.api.asforms.exceptions.UnsupportedFieldTypeException;
import kz.arta.synergy.api.asforms.pojo.*;
import org.codehaus.jackson.type.TypeReference;

import java.io.IOException;
import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;

/**
 * @author raimbek
 * @since 09.11.2016
 */
public class AsFormProvider {

    private RestHttpQuery restHttpQuery;

    private AsFormProvider(QueryContext context) {
        this.restHttpQuery = new RestHttpQuery(context);
    }

    public static AsFormProvider newInstance(QueryContext context) {
        return new AsFormProvider(context);
    }

    public List<AdvancedSearchResult> advancedSearch(AdvancedSearchParams searchParams) throws IOException {
        Query query = Query.newInstance()
                .methodPost()
                .header("Content-Type", "application/json;charset=utf-8")
                .url("/rest/api/asforms/search/advanced")
                .body(JsonUtils.toJson(searchParams));

        String result = restHttpQuery.doQuery(query);
        return JsonUtils.read(result, new TypeReference<List<AdvancedSearchResult>>() {});
    }

    public <T extends AsForm> T fetch(Class<T> formClass, String dataUUID) throws IOException {
        String data = restHttpQuery.doQuery(Query.newInstance().url("/rest/api/asforms/data/" + dataUUID));
        return create(formClass, JsonUtils.read(data, AsFormWrapper.class));
    }

    public <T extends AsForm> T create(Class<T> asFormClass, AsFormWrapper asfData) {
        T asForm = createAsFormObject(asFormClass, asfData, null);
        asForm.setForm(asfData.getForm());
        asForm.setModified(asfData.getModified());
        asForm.setNodeUUID(asfData.getNodeUUID());
        asForm.setUuid(asfData.getUuid());
        return asForm;
    }

    public <T extends AsForm> String save(T asForm) throws IOException {
        AsFormWrapper asFormWrapper = toAsfData(asForm);

        Query query = Query.newInstance()
            .methodPost()
            .url("/rest/api/asforms/data/save")
            .formParam("formUUID", asForm.getForm())
            .formParam("uuid", asForm.getUuid())
            .formParam("data", "\"data\":" + JsonUtils.toJson(asFormWrapper.getData()));

        return restHttpQuery.doQuery(query);
    }

    public <T extends AsForm> AsFormWrapper toAsfData(T asForm) {
        AsFormWrapper asFormWrapper = new AsFormWrapper();
        asFormWrapper.setForm(asForm.getForm());
        asFormWrapper.setModified(asForm.getModified());
        asFormWrapper.setNodeUUID(asForm.getNodeUUID());
        asFormWrapper.setUuid(asForm.getUuid());
        asFormWrapper.setData(toAsfData(asForm, null).getData());
        return asFormWrapper;
    }

    private <T> AsFormDataContainer toAsfData(T asForm, String index) {
        AsFormDataContainer dataContainer = new AsFormDataContainer();
        try {
            Field[] allFields = asForm.getClass().getDeclaredFields();
            for (Field field : allFields) {
                field.setAccessible(true);
                Object o = field.get(asForm);
                if (o == null) {
                    continue;
                }
                Annotation[] declaredAnnotations = field.getDeclaredAnnotations();
                for (Annotation annotation : declaredAnnotations) {
                    if (annotation instanceof TextBox) {
                        String cmpId = ((TextBox) annotation).value();
                        dataContainer.addData(toAsfData(asForm, field, cmpId, index, ComponentTypes.TEXT_BOX));
                        break;
                    }
                    if (annotation instanceof Cmp) {
                        Cmp cmpAnnotation = (Cmp) annotation;
                        dataContainer.addData(toAsfData(asForm, field, cmpAnnotation.id(), index, cmpAnnotation.type()));
                        break;
                    }
                    if (annotation instanceof NumericInput) {
                        String cmpId = ((NumericInput) annotation).value();
                        dataContainer.addData(toAsfData(asForm, field, cmpId, index, ComponentTypes.NUMERIC_INPUT));
                        break;
                    }
                    if (annotation instanceof Entity) {
                        String cmpId = ((Entity) annotation).value();
                        dataContainer.addData(toAsfData(asForm, field, cmpId, index, ComponentTypes.ENTITY));
                        break;
                    }
                    if (annotation instanceof ListBox) {
                        String cmpId = ((ListBox) annotation).value();
                        dataContainer.addData(toAsfData(asForm, field, cmpId, index, ComponentTypes.LISTBOX));
                        break;
                    }
                    if (annotation instanceof Table) {
                        Table tableAnnotation = (Table) annotation;
                        if (field.getType().isAssignableFrom(AsFormData.class)) {
                            dataContainer.addData(toAsfData(asForm, field, tableAnnotation.value(), index, ComponentTypes.TABLE));
                        } else if (field.getType().isAssignableFrom(List.class)) {
                            AsFormData tableData = listToAppendableTable(asForm, field, tableAnnotation.id());
                            dataContainer.addData(tableData);
                        } else {
                            throw new UnsupportedFieldTypeException();
                        }

                        break;
                    }
                }
            }
        } catch (IllegalAccessException e) {
            throw new CreateAsFormException("Form class should be accessible");
        }
        return dataContainer;
    }

    private <T> AsFormData toAsfData(T asForm, Field field, String cmpId, String index, String type) throws IllegalAccessException {
        field.setAccessible(true);

        if (index != null) {
            cmpId = cmpId + "-b" + index;
        }
        AsFormData asFormData = AsFormData.create(cmpId, type);

        if (ComponentTypes.TABLE.equals(type)) {
            AsFormData tableData = (AsFormData) field.get(asForm);
            if (tableData != null && tableData.getData() != null) {
                asFormData.setData(tableData.getData());
            }
        }

        Object valueObject = field.get(asForm);
        if (valueObject instanceof AsFormData) {
            AsFormData asFormDataObject = (AsFormData) valueObject;
            asFormDataObject.setId(cmpId);
            asFormDataObject.setType(type);
            return asFormDataObject;
        } else {
            if (type.equals(ComponentTypes.NUMERIC_INPUT)) {
                asFormData.setKey(String.valueOf(valueObject));
            }
            asFormData.setValue(String.valueOf(valueObject));
        }
        return asFormData;
    }

    private <T> AsFormData listToAppendableTable(T asForm, Field field, String id) throws IllegalAccessException {
        AsFormData tableData = AsFormData.create(id, ComponentTypes.TABLE);
        field.setAccessible(true);
        List tableList = (List) field.get(asForm);
        for (int bIndex = 1; bIndex <= tableList.size(); bIndex++) {
            AsFormDataContainer asFormDataContainer = toAsfData(tableList.get(bIndex - 1), String.valueOf(bIndex));
            for (AsFormData asFormData : asFormDataContainer.getData()) {
                tableData.addData(asFormData);
            }
        }
        return tableData;
    }

    private <T> T createAsFormObject(Class<T> asFormClass, AsFormDataContainer asfData, String index) {
        try {
            T asFormObject = asFormClass.newInstance();
            if (asfData == null || asfData.getData() == null || asfData.getData().isEmpty()) {
                return asFormObject;
            }

            Field[] allFields = asFormClass.getDeclaredFields();
            for (Field field : allFields) {
                Annotation[] declaredAnnotations = field.getDeclaredAnnotations();
                for (Annotation annotation : declaredAnnotations) {
                    if (annotation instanceof TextBox) {
                        String cmpId = ((TextBox) annotation).value();
                        setFieldValue(asfData, asFormObject, field, cmpId, index, ComponentTypes.TEXT_BOX);
                        break;
                    }
                    if (annotation instanceof Cmp) {
                        Cmp cmpAnnotation = (Cmp) annotation;
                        setFieldValue(asfData, asFormObject, field, cmpAnnotation.id(), index, cmpAnnotation.type());
                        break;
                    }
                    if (annotation instanceof NumericInput) {
                        String cmpId = ((NumericInput) annotation).value();
                        setFieldValue(asfData, asFormObject, field, cmpId, index, ComponentTypes.NUMERIC_INPUT);
                        break;
                    }
                    if (annotation instanceof Entity) {
                        String cmpId = ((Entity) annotation).value();
                        setFieldValue(asfData, asFormObject, field, cmpId, index, ComponentTypes.ENTITY);
                        break;
                    }
                    if (annotation instanceof ListBox) {
                        String cmpId = ((ListBox) annotation).value();
                        setFieldValue(asfData, asFormObject, field, cmpId, index, ComponentTypes.LISTBOX);
                        break;
                    }
                    if (annotation instanceof Table) {
                        Table tableAnnotation = (Table) annotation;
                        if (field.getType().isAssignableFrom(AsFormData.class)) {
                            setFieldValue(asfData, asFormObject, field, tableAnnotation.value(), index, ComponentTypes.TABLE);
                        } else if (field.getType().isAssignableFrom(List.class)) {
                            List list = tableToList(asfData, tableAnnotation.type(), tableAnnotation.id());
                            field.setAccessible(true);
                            field.set(asFormObject, list);
                        } else {
                            throw new UnsupportedFieldTypeException();
                        }

                        break;
                    }
                }
            }
            return asFormObject;
        } catch (InstantiationException e) {
            throw new CreateAsFormException("Form class should has accessible without params constructor");
        } catch (IllegalAccessException e) {
            throw new CreateAsFormException("Form class should be accessible");
        }
    }

    private <T> void setFieldValue(
            AsFormDataContainer asfData,
            T asFormObject,
            Field field,
            String cmpId,
            String index,
            String cmpType
    ) throws IllegalAccessException {
        field.setAccessible(true);

        if (index != null) {
            cmpId = cmpId + "-b" + index;
        }

        if (field.getType().isAssignableFrom(String.class)) {
            // string
            String value = getValueForClassField(field, asfData, cmpId);
            field.set(asFormObject, value);

        } else if (field.getType().isAssignableFrom(Integer.class)) {
            // int
            String value = getValueForClassField(field, asfData, cmpId);
            if (value == null) {
                value = "0";
            }
            field.set(asFormObject, Integer.parseInt(value));

        } else if (field.getType().isAssignableFrom(Double.class)) {
            // double
            String value = getValueForClassField(field, asfData, cmpId);
            if (value == null) {
                value = "0";
            }
            field.set(asFormObject, Double.parseDouble(value));

        } else if (field.getType().isAssignableFrom(AsFormData.class)) {
            // common type
            AsFormData data = asfData.getData(cmpId);
            if (data != null) {
                AsFormData fieldAsFormData = new AsFormData();
                fieldAsFormData.setId(cmpId);
                fieldAsFormData.setType(cmpType);
                fieldAsFormData.setValue(data.getValue());
                fieldAsFormData.setKey(data.getKey());
                fieldAsFormData.setKeys(data.getKeys());
                fieldAsFormData.setValues(data.getValues());
                fieldAsFormData.setUserID(data.getUserID());
                fieldAsFormData.setValueID(data.getValueID());
                fieldAsFormData.setLabel(data.getLabel());
                fieldAsFormData.setUsername(data.getUsername());
                fieldAsFormData.setData(data.getData());

                field.set(asFormObject, fieldAsFormData);
            }
        } else {
            throw new UnsupportedFieldTypeException("This type unsupported");
        }
    }

    @SuppressWarnings("unchecked")
    private List tableToList(AsFormDataContainer asfData, Class genericType, String cmpId) {
        AsFormData tableData = asfData.getData(cmpId);
        if (tableData == null || tableData.getData() == null || tableData.getData().isEmpty()) {
            return null;
        }

        List table = new ArrayList<>();
        Set<String> indexes = AsfTableUtil.tableBIndexes(tableData);
        for (String index : indexes) {
            table.add(createAsFormObject(genericType, tableData, index));
        }
        return table;
    }

    private String getValueForClassField(Field field, AsFormDataContainer asfData, String cmpId) {
        if (hasKeyValueAnnotation(field)) {
            return asfData.getKey(cmpId);
        }
        return asfData.getValue(cmpId);
    }

    private boolean hasKeyValueAnnotation(Field field) {
        boolean fetchKey = false;
        Annotation[] declaredAnnotations = field.getDeclaredAnnotations();
        for (Annotation annotation : declaredAnnotations) {
            if (annotation instanceof KeyValue) {
                fetchKey = true;
                break;
            }
        }
        return fetchKey;
    }

    public RestHttpQuery getRestHttpQuery() {
        return restHttpQuery;
    }

    public void setRestHttpQuery(RestHttpQuery restHttpQuery) {
        this.restHttpQuery = restHttpQuery;
    }
}
