package kz.arta.synergy.api.services;

import kz.arta.synergy.api.JsonUtils;
import kz.arta.synergy.api.Query;
import kz.arta.synergy.api.QueryContext;
import kz.arta.synergy.api.RestHttpQuery;
import kz.arta.synergy.api.pojo.SynergyFile;
import kz.arta.synergy.api.services.storage.Attachment;
import kz.arta.synergy.api.services.storage.AttachmentList;
import org.codehaus.jackson.type.TypeReference;

import java.io.IOException;
import java.util.List;

/**
 * Created by: Abzal Kultayev
 * Date: 12.11.16
 * Time: 16:37
 */
public class StorageService {
    private final RestHttpQuery restHttpQuery;

    private StorageService(QueryContext context) {
        this.restHttpQuery = new RestHttpQuery(context);
    }

    public static StorageService newInstance(QueryContext context) {
        return new StorageService(context);
    }

    /**
     * Создает временный файл на сервере
     * @return Возвращает путь временного файла на сервере
     * @throws IOException
     */
    public String startUploadFile() throws IOException {
        Query query = Query.newInstance().url("/rest/api/storage/start_upload");
        String result = restHttpQuery.doQuery(query);
        return JsonUtils.getValueByKey(result, "file");
    }


    /**
     * Метод добавляет вложение к документу из временного файла, загруженного на сервер с помощью API метода rest/api/storage/start_upload
     * @param documentID идентификатор документа
     * @param filename название файла
     * @param filepath путь до файла на сервере
     * @return errorCode и errorMessage
     * @throws IOException при проблеммах с сетью
     */
    public String createAttachment(String documentID, String filename, String filepath) throws IOException {
        Query query = Query.newInstance()
                .methodPost()
                .url("/rest/api/docflow/doc/attachment/create")
                .formParam("fileName", filename)
                .formParam("filePath", filepath)
                .formParam("path", "ase:workContainer")
                .formParam("documentID", documentID);
        return restHttpQuery.doQuery(query);
    }

    public Attachment createAndGetAttachment(String documentID, String filename, String filepath) throws IOException {
        String attachment = createAttachment(documentID, filename, filepath);
        if (!JsonUtils.getValueByKey(attachment, "errorCode").equals("0")) {
            throw new IOException("Attachment was not created");
        }
        AttachmentList lastAttachments = getLastAttachments(documentID);
        return lastAttachments.get(0);
    }

    public AttachmentList getLastAttachments(String documentID) throws IOException {
        Query query = Query.newInstance()
                .url("/rest/api/docflow/doc/last_attachments")
                .queryParam("documentID", documentID);

        return JsonUtils.read(restHttpQuery.doQuery(query), AttachmentList.class);
    }

    /**
     * Получит список файлов в папке
     * @param parentID  id родительской папки, если не передано,
     *                  будет возвращено корневые папки
     * @return  список файлов
     * @throws IOException
     */
    public List<SynergyFile> storageList(String parentID) throws IOException {
        Query query = Query.newInstance()
                .url("/rest/api/storage/list");
        if (parentID != null) {
            query.queryParam("parentID", parentID);
        }
        return JsonUtils.read(restHttpQuery.doQuery(query), new TypeReference<List<SynergyFile>>() {});
    }

    public SynergyFile findInStorage(String parentId, String fileName) throws IOException {
        List<SynergyFile> synergyFiles = storageList(parentId);
        for (SynergyFile synergyFile : synergyFiles) {
            if (synergyFile.getName().equals(fileName)) {
                return synergyFile;
            }
            if (synergyFile.isFolder()) {
                SynergyFile result = findInStorage(synergyFile.getIdentifier(), fileName);
                if (result != null) {
                    return result;
                }
            }
        }
        return null;
    }

    public String copy(String identifier, String documentId) throws IOException {
        Query query = Query.newInstance()
                .methodPost()
                .url("/rest/api/storage/copy")
                .formParam("fileID", identifier)
                .formParam("documentID", documentId);
        String result = restHttpQuery.doQuery(query);
        return JsonUtils.getValueByKey(result, "fileID");
    }

    public String add(String identifier, String tmpFileName, String synergyFileName) throws IOException {
        Query query = Query.newInstance().url("/rest/api/storage/add")
                .methodPost()
                .formParam("folderID", identifier)
                .formParam("path", tmpFileName)
                .formParam("name", synergyFileName);
        String result = restHttpQuery.doQuery(query);
        return JsonUtils.getValueByKey(result, "fileID");
    }

    public String move(String fileIdentifier, String folderID) throws IOException {
        Query query = Query.newInstance()
                .methodPost()
                .url("/rest/api/storage/move")
                .formParam("elementID", fileIdentifier)
                .formParam("folderID", folderID);
        return restHttpQuery.doQuery(query);
    }

    public String createFolder(String parent, String name) throws IOException {
        Query query = Query.newInstance()
                .methodPost()
                .url("/rest/api/storage/create")
                .formParam("folderID", parent)
                .formParam("folderName", name);
        return restHttpQuery.doQuery(query);
    }

    public byte[] downloadFile(String identifier) throws IOException {
        Query query = Query.newInstance()
                .url("/rest/api/storage/file/get")
                .queryParam("identifier", identifier)
                .queryParam("inline", "true");
        return restHttpQuery.doQueryAndReturnBytes(query);
    }
}
