package kz.arta.synergy.api.asforms.converter.components;

import kz.arta.synergy.api.asforms.annotations.KeyValue;
import kz.arta.synergy.api.asforms.exceptions.UnsupportedFieldTypeException;
import kz.arta.synergy.api.asforms.pojo.AsFormData;
import kz.arta.synergy.api.asforms.pojo.AsFormDataContainer;
import kz.arta.synergy.api.asforms.utils.Strings;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.math.BigDecimal;
import java.math.BigInteger;

/**
 * @author raimbek
 * @since 11.11.2016
 */
public abstract class AbstractComponentConverter implements ComponentConverter {

    @Override
    public <T> AsFormData toAsfData(T asForm, Field field, Annotation annotation, String index) throws IllegalAccessException {
        field.setAccessible(true);
        String cmpId = getCmpId(field, annotation, index);

        AsFormData asFormData = AsFormData.create(cmpId, getType(annotation));

        Object valueObject = field.get(asForm);
        if (valueObject instanceof AsFormData) {
            AsFormData asFormDataObject = (AsFormData) valueObject;
            asFormDataObject.setId(cmpId);
            asFormDataObject.setType(getType(annotation));
            return asFormDataObject;
        } else {
            asFormData.setValue(String.valueOf(valueObject));
        }
        return asFormData;
    }

    @Override
    public <T> void setFieldValue(AsFormDataContainer asfData, T asFormObject, Field field, Annotation annotation, String index) throws Exception {
        field.setAccessible(true);
        String cmpId = getCmpId(field, annotation, index);

        if (field.getType().isAssignableFrom(String.class)) {
            // string
            String value = getValueForClassField(field, asfData, cmpId);
            field.set(asFormObject, value);

        } else if (field.getType().isAssignableFrom(Integer.class)) {
            // int
            String value = getNumberValueForField(asfData, field, cmpId);
            field.set(asFormObject, Integer.parseInt(value));

        } else if (field.getType().isAssignableFrom(BigInteger.class)) {
            // int
            String value = getNumberValueForField(asfData, field, cmpId);
            field.set(asFormObject, new BigInteger(value));

        } else if (field.getType().isAssignableFrom(Double.class)) {
            // double
            String value = getNumberValueForField(asfData, field, cmpId);
            value = value.replaceAll(",", ".");
            field.set(asFormObject, Double.parseDouble(value));

        } else if (field.getType().isAssignableFrom(BigDecimal.class)) {
            // double
            String value = getNumberValueForField(asfData, field, cmpId);
            field.set(asFormObject, new BigDecimal(value));

        } else if (field.getType().isAssignableFrom(AsFormData.class)) {
            // common type
            AsFormData data = asfData.getData(cmpId);
            AsFormData fieldAsFormData = new AsFormData();
            if (data != null) {
                fieldAsFormData.setId(cmpId);
                fieldAsFormData.setType(getType(annotation));
                fieldAsFormData.setValue(data.getValue());
                fieldAsFormData.setKey(data.getKey());
                fieldAsFormData.setKeys(data.getKeys());
                fieldAsFormData.setValues(data.getValues());
                fieldAsFormData.setUserID(data.getUserID());
                fieldAsFormData.setValueID(data.getValueID());
                fieldAsFormData.setLabel(data.getLabel());
                fieldAsFormData.setUsername(data.getUsername());
                fieldAsFormData.setData(data.getData());
            }
            field.set(asFormObject, fieldAsFormData);
        } else {
            throw new UnsupportedFieldTypeException("This type unsupported");
        }
    }

    protected String getCmpId(Field field, Annotation annotation, String index) {
        String cmpId = getCmpId(annotation);

        // если id компонента пустой то используем
        // название поле в качестве id компонента
        if (Strings.isNullOrEmpty(cmpId)) {
            cmpId = field.getName();
        }

        // учитываем случай с динамической таблицой
        if (index != null) {
            cmpId = cmpId + "-b" + index;
        }
        return cmpId;
    }

    private String getValueForClassField(Field field, AsFormDataContainer asfData, String cmpId) {
        if (hasKeyValueAnnotation(field)) {
            return asfData.getKeyFromInnerData(cmpId);
        }
        return asfData.getValueFromInnerData(cmpId);
    }

    private String getNumberValueForField(AsFormDataContainer asfData, Field field, String cmpId) {
        String value = getValueForClassField(field, asfData, cmpId);
        if (Strings.isNullOrEmpty(value)) {
            value = "0";
        }
        return value.trim().replaceAll(" ", "");
    }

    protected boolean hasKeyValueAnnotation(Field field) {
        boolean fetchKey = false;
        Annotation[] declaredAnnotations = field.getDeclaredAnnotations();
        for (Annotation annotation : declaredAnnotations) {
            if (annotation instanceof KeyValue) {
                fetchKey = true;
                break;
            }
        }
        return fetchKey;
    }
}
