package kz.arta.synergy.api.dictionaries;

import kz.arta.synergy.api.JsonUtils;
import kz.arta.synergy.api.Query;
import kz.arta.synergy.api.QueryContext;
import kz.arta.synergy.api.RestHttpQuery;
import kz.arta.synergy.api.asforms.utils.Strings;
import kz.arta.synergy.api.dictionaries.annotations.DictionaryCode;
import kz.arta.synergy.api.dictionaries.annotations.DictionaryColumn;
import kz.arta.synergy.api.dictionaries.pojo.Dictionary;
import kz.arta.synergy.api.dictionaries.pojo.DictionaryItem;
import kz.arta.synergy.api.dictionaries.pojo.DictionaryItemValue;

import java.io.IOException;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author raimbek
 * @since 15.11.2016
 */
public class DictionaryService {

    private final RestHttpQuery restHttpQuery;

    private DictionaryService(QueryContext context) {
        this.restHttpQuery = new RestHttpQuery(context);
    }

    public static DictionaryService newInstance(QueryContext context) {
        return new DictionaryService(context);
    }

    public Dictionary getDictionary(String code) throws IOException {
        Query query = Query.newInstance()
                .url("/rest/api/dictionary/get_by_code")
                .queryParam("dictionaryCode", code);

        return JsonUtils.read(restHttpQuery.doQuery(query), Dictionary.class);
    }

    public <T> List<T> getDictionary(Class<T> dictionaryType) throws DictionaryConvertException {
        try {
            ArrayList<T> dictionaryList = new ArrayList<>();

            DictionaryCode dictionaryCodeAnnotation = dictionaryType.getAnnotation(DictionaryCode.class);
            if (dictionaryCodeAnnotation == null) {
                throw new DictionaryConvertException(
                        "Dictionary class should be marked tiwh annotation " + DictionaryCode.class.getName()
                );
            }

            Dictionary dictionary = getDictionary(dictionaryCodeAnnotation.value());

            Map<String, Field> columnIds = new HashMap<>();
            for (Field classField : dictionaryType.getDeclaredFields()) {
                DictionaryColumn dictionaryColumnAnnotation = classField.getAnnotation(DictionaryColumn.class);
                if (dictionaryColumnAnnotation == null) {
                    continue;
                }

                String columnCode = dictionaryColumnAnnotation.value();
                if (Strings.isNullOrEmpty(columnCode)) {
                    columnCode = classField.getName();
                }

                String columnIdByCode = dictionary.findColumnIdByCode(columnCode);
                if (columnIdByCode != null) {
                    classField.setAccessible(true);
                    columnIds.put(columnIdByCode, classField);
                }
            }

            for (DictionaryItem dictionaryItem : dictionary.getItems()) {
                T dictionaryItemObject = dictionaryType.newInstance();
                for (DictionaryItemValue itemValue : dictionaryItem.getValues()) {
                    if (columnIds.containsKey(itemValue.getColumnID())) {
                        Field field = columnIds.get(itemValue.getColumnID());
                        field.set(dictionaryItemObject, itemValue.getValue());
                    }
                }
                dictionaryList.add(dictionaryItemObject);
            }

            return dictionaryList;
        } catch (Exception ex) {
            throw new DictionaryConvertException(ex);
        }
    }
}
