#!/bin/bash
#
# This script is for building changed documents on
# TDD's jenkins.
# 
# by Sergey Korobitsin <undertaker@arta.kz>, 
#    Raimbek Egemberdiev <r.egemberdiev@arta.pro>
# 

set -e

export PATH="/usr/local/bin:/usr/bin:/bin"

## CONFIG SECTION

# FTP Root
FTPROOT="/srv/docs/product"

# Files group
GROUP="cddd"

# Directories with documentation to make
DOC_DIRS="
mobile-spec
crm/methodologist-manual
crm/spec
crm/user-manual
crm/additional
hrm/developer-manual
hrm/spec
hrm/user-manual
hrm/additional
"

FULL_DOC_DIRS="
admin-manual
methodologist-manual
user-manual
"

# Targets of documentation to make
TARGETS="html"
FULL_TARGETS="html-full pdf-full"

# Additional targets for overwrite default targets
declare -A ADDITIONAL_TARGETS
ADDITIONAL_TARGETS["crm/methodologist-manual"]="html pdf"
ADDITIONAL_TARGETS["crm/user-manual"]="html pdf"
ADDITIONAL_TARGETS["crm/spec"]="html pdf docx"
ADDITIONAL_TARGETS["hrm/developer-manual"]="html pdf"
ADDITIONAL_TARGETS["hrm/user-manual"]="html pdf"
ADDITIONAL_TARGETS["hrm/spec"]="html pdf docx"

## END CONFIG SECTION

get_build_dir() {
    local orig_path
    local result_path

    _get_build_dir() {
        local cpath="$1"

        if [ ! -e "$cpath" ]; then
            echo "WARN: $cpath does not exist" >&2
        fi
        if [ -f "$cpath" ]; then
            # if argument is file
            if [ "$(basename $cpath)" = 'Makefile' ]; then
                echo "$(dirname $cpath)"
                return 0
            fi
        else
            # if argument is directory
            if [ -f "$cpath/Makefile" ]; then
                echo "$cpath"
                return 0
            fi
            if [ "$cpath" = '.' ]; then
                echo "$cpath"
                return 0
            fi
        fi
        # Trying to get dir recursively
        _get_build_dir "$(dirname $cpath)"
    }

    while read orig_path; do
        result_path="$(_get_build_dir $orig_path)"
        if [ "$result_path" = '.' ]; then
            echo "WARN: no directory with Makefile found for $orig_path" >&2
            echo "$orig_path"
        else
            echo "$result_path"
        fi
    done
}

# Check if we got proper parameters
REPOS="/svnroot/cddd"
CHANGEDBY="$1"

if [ -z "$GIT_COMMIT" ]; then
    echo "ERROR: environmental variable GIT_COMMIT not set" >&2
    exit 1
fi
if [ -z "$GIT_PREVIOUS_SUCCESSFUL_COMMIT" ]; then
    echo "WARN: environmental variable GIT_PREVIOUS_SUCCESSFUL_COMMIT not set, using GIT_COMMIT" >&2
    GIT_PREVIOUS_SUCCESSFUL_COMMIT="$GIT_COMMIT"
fi

GIT_BRANCH=$(git symbolic-ref --short HEAD)

case "$GIT_BRANCH" in
    master)
        FTPROOT="$FTPROOT/"
    ;;
    *)
        FTPROOT="$FTPROOT/tags/$GIT_BRANCH"
    ;;
esac

# Setting up TMPDIR
[ -n "$TMPDIR" ] || TMPDIR="/tmp"

git diff --name-only $GIT_PREVIOUS_SUCCESSFUL_COMMIT $GIT_COMMIT | \
    get_build_dir | sort -u | \
    while read changed_dir; do
        if echo "$DOC_DIRS" | grep -q "^$changed_dir$"; then
            changed_commit=$(git log -1 --pretty=format:%h $changed_dir)
            changed_by=$(git log -1 --pretty=format:%an $changed_dir)
            echo "Building $changed_dir, commit $changed_commit by $changed_by"
            make -C "$changed_dir" clean

            FINAL_TARGETS=$TARGETS
            if echo "$FULL_DOC_DIRS" | grep -q "^$changed_dir"; then
                FINAL_TARGETS=$FULL_TARGETS
            fi
            # Kinda too much targets
            # and they are not additional anyway
            # TODO: refactor this
            [ ${ADDITIONAL_TARGETS["$changed_dir"]+x} ] && TARGETS=${ADDITIONAL_TARGETS["$changed_dir"]}
            
            make VCSCHANGEDBY="$changed_by" VCSREV="$changed_commit" -C "$changed_dir" $FINAL_TARGETS

            # Move everything that has been built to ftp
            # Special treatment - look into top level and remove destination
            # if it exists before copying
            echo "Deploying $changed_dir to FTP"
            mkdir -p "${FTPROOT}/${changed_dir}"
            for bin in "$changed_dir"/bin/*; do
                [ -e "${FTPROOT}/${changed_dir}/$(basename $bin)" ] && rm -rv "${FTPROOT}/${changed_dir}/$(basename $bin)"
                cp -rv "$bin" "${FTPROOT}/${changed_dir}"
            done
        else
            echo "$changed_dir is not in DOC_DIRS, skipping"
        fi
    done
